' Reversi2 for CMM2.
' Rev 1.0.0 William M Leue 17-June-2022

option default integer
option base 0

' Constants
const MCHAN = 2 ' mouse I2C channel; set to zero if no mouse

const SIZE = 8
const GRAPHICS = 0
const DEBUG = 1

const MAX_DEPTH = 16
const ENDGAME_NPIECES = SIZE*SIZE - MAX_DEPTH

const PBLACK = 1
const PWHITE = -1
const FWHITE = 4  ' For packing into integer
const EMPTY = 0
const OUT = 3
const NUM_DIRECTIONS = 8

const STACK_SIZE = 200
const MAX_MOVES = 40

' Game Statuses
const GAME_RUNNING = 0
const COMPUTER_WINS = 1
const PLAYER_WINS = 2
const TIE_GAME = 3
const NO_PLAYER_MOVES = 4
const NO_COMPUTER_MOVES = 5
const NO_MOVE = 0

const BWIDTH = 20
const BOARDCOLOR = RGB(0, 130, 30)
const BCOLOR = RGB(GRAY)
const CBLACK = RGB(BLACK)
const CWHITE = RGB(WHITE)
const CELLSIZE = 40
const PIECERADIUS = 0.40*CELLSIZE
const LEFTMARGIN = 30
const TOPMARGIN = 60
const STATUS_WIDTH =  500
const STATUS_HEIGHT = 50
const STATUS_X = 50
const STATUS_Y = 460
const POLICY_X = 400
const POLICY_Y = 20
const OCCUPIED = 17
const BIG_VALUE = 10000
const FACE1 = 135
const FACE2 = 136

' Mouse Menu
const NMENUS = 3
const MW = 60
const MH = 20
const MY = 410
const MX = 30
const MSPAC = 20
const MNEW = -1
const MGO = -2
const MQUIT = -3

' Keyboard inputs
const UP    = 128
const DOWN  = 129
const LEFT  = 130
const RIGHT = 131
const ENTER = 13
const F12   = 156
const KEND  = 135
const QUEST = 63
const ESC   = 27

' Global Variables
dim board(SIZE+2, SIZE+2)
dim offsets(8, 2)
dim stack(STACK_SIZE)
dim current_depth = INITIAL_DEPTH
dim stackPtr = 0
dim player = PBLACK
dim computer = PWHITE
dim gameRunning = 0
dim pChosenMove = 1

dim float policyweight1
dim float policyweight2
dim float policyweight3
dim float policyweight4
dim float policyweight5

dim numCMoves = 0
dim edge_vals(SIZE) = (0, 10, 0, 1, 1, 1, 1, 0, 10)

dim BoardWeights(SIZE, SIZE)
dim UpdateParams(4, 12)

dim float xscale = 1.0
dim float yscale = 1.0
dim xoff = 0
dim yoff = 0
dim menuX(NMENUS)

' mouse handshake variables
dim hasMouse = 0
dim mouseX = 0
dim mouseY = 0
dim left_click = 0

' keyboard control variables
dim kb_cursor_col = 0
dim kb_cursor_row = 0

' performance data collection
dim float move_times(32)
dim float mtp = 0

' endgame transition
dim isEndGame = 0

' Main program
open "debug.txt" for output as #1
InitMouse
SetOffsets
NewGame
DrawScreen
if onTheMove = player then
  ShowMoves player
end if
if hasMouse then
  PlayLoopMouse
else
  PlayLoopKeyboard
end if
end

' See if a mouse is available
sub InitMouse
  local x, y
  on error skip 1
  controller mouse open MCHAN, LeftClick
  if MM.ERRNO <> 0 then
    hasMouse = 0
    gui cursor on 1, 0, 0, rgb(red)
    kb_cursor_col = 4
    kb_cursor_row = 4
    SetKBCursor    
    exit sub
  else
    hasMouse = 1
    settick 16, UpdateCursor
    gui cursor on 1, 0, 0, rgb(red)
    gui cursor show
  end if
  pause 20
end sub

' make cursor track the mouse
sub UpdateCursor
  gui cursor mouse(x, MCHAN), mouse(y, MCHAN)
end sub

' Interrupt Service for Left Mouse Click
' Handle player moves and menu choices.
sub LeftClick
  if left_click = 0 then
    mouseX = mouse(X) : mouseY = mouse(Y)
    left_click = 1
  end if
end sub

' Play loop when using a mouse
sub PlayLoopMouse
  local status, which, ok
  local z$, cmd
  local availableMoves(MAX_MOVES)
  z$ = INKEY$
  do
    z$ = INKEY$
    if z$ <> "" then
      cmd = asc(UCASE$(z$))
      select case cmd
        case QUEST
          ShowHelp
        case ESC
          Quit
      end select
    end if
    if gameRunning then
      status = GetGameStatus()
    end if
    if left_click then
      which = GetClickLocation(mouseX, mouseY)
      if which < 0 then
        HandleMenuSelection which
      end if
    end if
    if not gameRunning then continue do
    if onTheMove = player then
      GetMoves player, availableMoves()
      if availableMoves(0) = 0 then
        onTheMove = computer
        continue do
      end if
      ShowMoves player
      if left_click then
        which = GetClickLocation(mouseX, mouseY)
        if which = 0 then
          continue do
        else if which < 0 then
          HandleMenuSelection which
        else
          ok = HandlePlayerMove(which)
          if ok then
            status = GetGameStatus()
            if status <> GAME_RUNNING then continue do
            onTheMove = computer
          end if
        end if
        left_click = 0
      end if
    end if
    if onTheMove = computer then
      status = GetGameStatus()
      if status <> GAME_RUNNING then continue do
      ok = HandleComputerMove()
      status = GetGameStatus()
      if status <> GAME_RUNNING then continue do
      if status <> NO_PLAYER_MOVES then
        onTheMove = player
      end if
    end if
  loop
end sub

' Play Loop when using Keyboard Only
sub PlayLoopKeyboard
  local status, which, ok
  do
    status = GetGameStatus()
    if status <> GAME_RUNNING then continue do
    if onTheMove = player then
      which = GetUserKBCommand()
      if which <> NO_COMMAND then
        ok = HandlePlayerMove(which)
        if ok then
          status = GetGameStatus()
          if status <> GAME_RUNNING then continue do
            onTheMove = computer
          end if
        end if
      end if
    end if
    if onTheMove = computer then
      status = GetGameStatus()
      if status <> GAME_RUNNING then continue do
      ok = HandleComputerMove()
      status = GetGameStatus()
      if status <> GAME_RUNNING then continue do
      onTheMove = player
    end if
  loop    
end sub

' Get User Keyboard Command
' This function handles all the menu commands, but
' passes moves to its caller.
function GetUserKBCommand()
  local z$, cmd, user_command
  user_command = NO_COMMAND
  z$ = INKEY$
  do
    do
      z$ = INKEY$  
    loop until z$ <> ""
    cmd = asc(UCASE$(z$))
    select case cmd
      case UP
        if kb_cursor_row > 1 then 
          inc kb_cursor_row, -1
          SetKbCursor
        end if
      case DOWN
        if kb_cursor_row < SIZE then
         inc kb_cursor_row
         SetKbCursor
        end if
      case LEFT
        if kb_cursor_col > 1 then
          inc kb_cursor_col, -1
          SetKBCursor
        end if
      case RIGHT
        if kb_cursor_col < SIZE then
          inc kb_cursor_col
          SetKBCursor
        end if
      case ENTER
        user_command = kb_cursor_row*100 + kb_cursor_col
        exit do
      case F12
        NewGame
        DrawBoard
        'ShowPolicyDetails
        ShowMoves player
        DrawStatus "Your Move"
      case KEND
        ComputerMovesFirst
      case QUEST
        ShowHelp
      case ESC
        Quit
    end select
  loop
  GetUserKBCommand = user_command
end function

' Move the cursor around based on keyboard inputs
sub SetKBCursor
  local x, y
  x = LEFTMARGIN + (kb_cursor_col-1)*CELLSIZE + CELLSIZE\2
  y = TOPMARGIN + (kb_cursor_row-1)*CELLSIZE + CELLSIZE\2
  gui cursor x, y
end sub

' Handle Menu Selections (mouse only)
sub HandleMenuSelection which
  select case which
    case MNEW
      NewGame
      DrawBoard
      ShowPolicyDetails
      ShowMoves player
      DrawStatus "Your Move"
    case MGO
      ComputerMovesFirst
    case MQUIT
      Quit()
  end select
end sub

' Quit the game
sub Quit
  if hasMouse then
    settick 0,0
    Controller Mouse close
  end if
  close #1
  cls
  end
end sub

' Handle Player Move
function HandlePlayerMove(which)
  local row, col, ok, i
  local avrow, avcol, bookmark
  local availableMoves(MAX_MOVES)
  row = which\100
  col = which - 100*row
  GetMoves player, availableMoves()
  ok = 0
  for i = 1 to availableMoves(0)
    UnpackMove availableMoves(i), avrow, avcol, player, bookmark
    if row = avrow and col = avcol then
      ok = 1
      exit for
    end if
  next i
  if ok then
    nrev = DoMove(row, col, player, 0)
    ClearStack
    UpdateBoardWeights
    onTheMove = computer
    DrawBoard
    HiliteLastMove row, col, player
    ShowMoves computer
  end if
  HandlePlayerMove = ok
end function

' Check Game Status
function GetGameStatus()
  local move, ok, i, status, full, npieces, numc, nump
  local availableMoves(MAX_MOVES)
  status = GAME_RUNNING
  full = 0
  npieces = getNumPieces(EMPTY)
  if npieces = 0 then full = 1
  numc = GetNumPieces(computer)
  nump = GetNumPieces(player)
  if not full then
    GetMoves computer,  availableMoves()
    nmovesc = availableMoves(0)
    GetMoves player,  availableMoves()
    nmovesp = availableMoves(0)
  end if
  if full or (nmovesc = 0 and nmovesp = 0) then
    if numc > nump then
      status = COMPUTER_WINS
      DrawStatus "Game Over -- Computer Wins, " + str$(numc) + " to " + str$(nump)
      gameRunning = 0
    else if nump > numc then
      status = PLAYER_WINS
      DrawStatus "Game Over -- You Win, " + str$(nump) + " to " + str$(numc)
      gameRunning = 0
    else
      status = TIE_GAME
      DrawStatus "Game Over -- A Tie, " + str$(numc) + " to " + str$(nump)
      gameRunning = 0
    end if
  end if
  if gameRunning then
    if nmovesp = 0 then
      status = NO_PLAYER_MOVES
      DrawStatus "You have no moves -- Computer moves again"
      ShowMoves computer
      pause 1000
    else if nmovesc = 0 then
      status = NO_COMPUTER_MOVES
      DrawStatus "The Computer has no moves -- You move again"
      ShowMoves player
    end if
  end if
  GetGateStatus = status
end function

' Handle Computer Move
' returns 1 if there was a legal move, 0 if not
function HandleComputerMove()
  local move, nrev, row, col, bookmark, status
  timer = 0
  status = GetGameStatus()
  DrawStatus "Computer Thinking..."
  pause 1000
  inc numCMoves
  move = GetBestMove(computer, current_depth)
  if move = -1 then
    HandleComputerMove = 0
    exit function
  end if
  UnpackMove move, row, col, computer, bookmark
  nrev = DoMove(row, col, computer, 0)
  ClearStack
  UpdateBoardWeights
  onTheMove = player
  DrawBoard
  HiliteLastMove row, col, computer
  status = GetGameStatus()
  if status <> GAME_RUNNING then exit sub
  if status <> NO_PLAYER_MOVES then
    ShowMoves player
    DrawStatus "Your Move"
  end if
  HandleComputerMove = 1
  move_times(numCMoves) = timer
  if numCMoves > 20 and move_times(numCMoves) < 1500.0 then
    inc current_depth
  end if
end function

' Get the Board Region where a left click occured
' > 0: on board: row*100 + col
' < 0: in command cell
' = 0: nothing
function GetClickLocation(x, y)
  local bx1, bx2, by1, by2, row, col
  bx1 = LEFTMARGIN : by1 = TOPMARGIN
  bx2 = bx1 + SIZE*CELLSIZE
  by2 = by1 + SIZE*CELLSIZE
  if x >= bx1 and x <= bx2 and y >= by1 and y <= by2 then
    row = (y-by1)\CELLSIZE + 1
    col = (x-bx1)\CELLSIZE + 1
    GetClickLocation = row*100 + col
    exit function
  else
    if y >= MY and y <= MY+MH then
      for i = 1 to NMENUS
        if x >= menuX(i) and x <= menuX(i) + MW then
          GetClickLocation = -i
          exit function
        end if
      next i
    end if
  else
    GetClickLocation = 0
  end if
end function

' Set up offset table for board searches
sub SetOffsets
  offsets(0, 0) = -1 : offsets(0, 1) = -1  
  offsets(1, 0) = -1 : offsets(1, 1) =  0  
  offsets(2, 0) = -1 : offsets(2, 1) =  1  
  offsets(3, 0) =  0 : offsets(3, 1) = -1  
  offsets(4, 0) =  0 : offsets(4, 1) =  1  
  offsets(5, 0) =  1 : offsets(5, 1) = -1  
  offsets(6, 0) =  1 : offsets(6, 1) =  0  
  offsets(7, 0) =  1 : offsets(7, 1) =  1
end sub

' Create a new game
' The player is set as the first player. Call the
' ComputerMovesFirst subroutine to have the computer
' move first instead. (first player is BLACK)
sub NewGame
  local row, col, i, j
  ' clear all pieces
  ZeroBoard
  ' Create the out-of-bounds codes
  for row = 0 to SIZE+1
    for col = 0 to SIZE+1
      if row = 0 or row = SIZE+1 then
        board(row, col) = OUT
      end if
    next col
    for col = 0 to SIZE+1
      if col = 0 or col = SIZE+1 then
        board(row, col) = OUT
      endif
    next col
  next row
  ' Insert the Starting Pieces
  board(4, 4) = PWHITE
  board(4, 5) = PBLACK
  board(5, 4) = PBLACK
  board(5, 5) = PWHITE
  ' Set up the board and policy weights
  restore weights
  for row = 1 to SIZE
    for col = 1 to SIZE
      read BoardWeights(row, col)
    next col
  next row  
  for i = 0 to 3
    for j = 0 to 11
      read UpdateParams(i, j)
    next j
  next i
  SetInitialPolicyWeights
  ' Set the default starting player
  ' the player can use the 'GO' command to have computer move first
  player = PBLACK
  onTheMove = player  
  if not hasMouse then
    kb_cursor_col = 4
    kb_cursor_row = 4
    SetKBCursor    
  end if
  stackPtr = 0
  numCMoves = 0
  isEndGame = 0
  current_depth = INITIAL_DEPTH
  gameRunning = 1
end sub

' Zero out Board
sub ZeroBoard
  local row, col
  for row = 1 to SIZE
    for col = 1 to SIZE
      board(row, col) = EMPTY
    next col
  next row
end sub

' set the initial policy weights
' Policy Weights for computer moves:
' 1: weighting for board values
' 2: weighting for number of pieces
' 3: weighting from lookahead
' These weights evolve as the game progresses.
sub SetInitialPolicyWeights
  policyWeight1 = 1
  policyWeight2 = 1
  policyWeight3 = 1
  policyWeight4 = 2
  policyWeight5 = 4
end sub

' Set up game where computer moves first (black)
sub ComputerMovesFirst
  np = GetNumPieces()
  if np > 4 then
    exit sub
  end if
  computer = PBLACK
  player = PWHITE
  onTheMove = computer
end sub

' Push a move onto the stack.
' Bookmark = 1: bookmark this move, = 0: don't
' After pushing the board is updated with the move.
sub PushMove(row, col, who, bookmark)
  local cell, bmove  
  cell = board(row, col)
  bmove = PackMove(row, col, cell, bookmark)
  stack(stackptr) = bmove
  stackptr = stackptr + 1
  board(row, col) = who
end sub

' Pop a single move or multiple moves from the stack onto the board
' multi = 1: pop moves to bookmark, 0: pop a single move
sub PopMove multi
  local row, col, who, bookmark, bmove, npop
  if multi = 1 then
    bookmark = 0
  else
    bookmark = 1
  end if
  npop = 0
  do
    stackptr = stackptr - 1
    bmove = stack(stackptr)
    UnpackMove bmove, row, col, who, bookmark
    board(row, col) = who
    npop = npop + 1
  loop until bookmark = 1
end sub

' Clear the stack by resetting the stack pointer to zero.
' This must be done to finalize a move.
sub ClearStack
  stackPtr = 0
end sub

' Pack the move components into a single integer
' (alias the negative PWHITE value with FWHITE)
function PackMove(row, col, who, bookmark) as integer
  local fplayer = who
  local move
  if who < 0 then
    fplayer = FWHITE
  end if
  move = row*1000 + col*100 + fplayer*10 + bookmark
  PackMove = move
end function

' Unpack a packed move into its components
' (Convert the FWHITE alias into the negative PWHITE)
sub UnpackMove move, row, col, who, bookmark
  row = move\1000
  col = (move - row*1000)\100
  who = (move - row*1000 - col*100)\10
  if who = FWHITE then who = PWHITE
  bookmark = move MOD 2
end sub

' Generate an array of legal moves from the current board position
' for the specified player. The zeroth element of the availableMoves
' array tells how many moves are available.
' This subroutine does not make any changes to the board.
sub GetMoves who, availableMoves()
  local row, col, nrev, index, amove
  local bookmark = 0
  index = 1
  for row = 1 to SIZE
    for col = 1 to SIZE
      nrev = 0
      if board(row, col) = EMPTY then
        nrev = TryMove(row, col, who)
        if nrev > 0 then
          amove = PackMove(row, col, who, 1) 
          availableMoves(index) = amove
          index = index + 1
        end if
      end if
    next col
  next row
  availableMoves(0) = index - 1
end sub

' Try a move and return the number of cells reversed,
' or zero if the move is illegal.
' This function does not make any changes to the board.
function TryMove(row, col, rplayer) as integer
  local nrev, trev, trow, tcol
  local cell, drow, dcol, index, dok
  nrev = 0
  for index = 0 to SIZE-1
    drow = offsets(index, 0)
    dcol = offsets(index, 1)
    trev = 0
    trow = row
    tcol = col
    dok = 0
    do
      inc trow, drow
      inc tcol, dcol
      cell = board(trow, tcol)
      select case cell
        case -rplayer
          trev = trev + 1
        case rplayer
          dok = 1
          exit do
        case else
          dok = 0
          trev = 0
          exit do
      end select
    loop
    if dok = 1 then
      nrev = nrev + trev
    end if
  next index
  TryMove = nrev
end function

' Execute a move. For performance reasons the move must be validated as legal
' before this function is used! The move is made on the board and all changed
' cells are pushed onto the stack. To make the move permanent, call the
' ClearStack subroutine. To cancel the move, call the PopMove subroutine with
' the bookmark parameter set to 1.
' The function returns the number of pieces flipped.      
function DoMove(row, col, rplayer, wflag) as integer
  local nrev, trev, trow, tcol, bookmark
  local cell, drow, dcol, dok, index, srow, scol, s

  if row <= 0 or col <= 0 or row > 8 or col > 8 then
    close #1
    print "BUG! row: " + str$(row) + " col: " + str$(col)
    exit
  end if
  if wflag = 0 then
    BoardWeights(row, col) = OCCUPIED
  end if
  bookmark = 1
  nrev = 0
  PushMove(row, col, rplayer, bookmark)
  for index = 0 to SIZE-1
    drow = offsets(index, 0)
    dcol = offsets(index, 1)
    trev = 0
    trow = row
    tcol = col
    dok = 0
    do
      trow = trow + drow
      tcol = tcol + dcol
      cell = board(trow, tcol)
        select case cell
        case -rplayer
          trev = trev + 1
        case rplayer
          srow = row
          scol = col
          if trev > 0 then
            for s = 1 to trev
              srow = srow + drow
              scol = scol + dcol
              PushMove(srow, scol, rplayer, 0)
            next s
            exit do
          end if
        case else
          trev = 0
          exit do
      end select
    loop
    nrev = nrev + trev
  next index
  DoMove = nrev
end function

' Return the number of pieces with the specified value.
' If the argument is not given, the total number of
' pieces of both colors will be returned.
function GetNumPieces(which) as integer
  local row, col, num, cell
  num = 0
  for row = 1 to SIZE
    for col = 1 to SIZE
      cell = board(row, col)
      if which = 0 then
        if cell <> 0 then inc num
      else
        if cell = which then inc num
      end if
    next col
  next row
  getNumPieces = num
end function

' Returns the 'best' move for the computer
' computer is the computer's piece color (1 or -1)
' player is the human's piece color (-1 or 1)
' Note: we don't check for an active game!
' returns the packed best move in an integer, or -1 if no moves
' This version does NOT do a lookahead: all moves are created by
' policy weights.
function GetBestMove (computer, depth) as integer
  local num_moves, i, theMove, row, col, alpha, beta
  local nrev, value, pvalue, nvalue, kvalue
  local bookmark = 0
  local bestMove = -1
  local bestValue = -BIG_VALUE   
  local availableMoves(MAX_MOVES)
  if GetNumPieces() = 5 then
    GetBestMove = MakePerpendicularMove()
    exit function
  end if
  getMoves computer,  availableMoves()
  num_moves = availableMoves(0)
  if num_moves = 0 then
    getBestMove = -1
    exit function
  end if
  value = 0
  bestValue = -BIG_VALUE
  for i=1 to num_moves
    theMove = availableMoves(i)
    UnpackMove(theMove, row, col, computer)
    pvalue = getPolicyValue(row, col, computer)
    value = policyWeight1*pvalue
    if bestMove < 0 or num_moves = 1 or value >= bestValue then
      bestMove = theMove
      bestValue = value
    end if
    DrawStatus "Computer Thinking " + chr$(FACE1 + (i MOD 2))
  next i
  getBestMove = bestMove
  UnpackMove(theMove, row, col, computer)
end function

' Make 'book' perpendicular move in response to player going first
function MakePerpendicularMove()
  local row, col
  if board(3,4) = player then
    row = 5 : col = 3
  else if board(4,3) = player then
    row = 3 : col = 5
  else if board(5,6) = player then
    row = 6 : col = 4
  else if board(6,5) = player then
    row = 4 : col = 6
  end if
  MakePerpendicularMove = PackMove(row, col, computer, 0)
end function

' Returns number of connected friendly neighbord
' (uses VonNeumann neighborhood)
function GetNumNeighbors(row, col)
  local d, drow, dcol, n
  n = 0
  for d = 1 to 4
    select case d
      case 1
        drow = row : dcol = col-1
      case 2
        drow = row+1 : dcol = col
      case 3
        drow = row : dcol = col+1
      case 4
        drow = row-1 : dcol = col
    end select
    if board(drow, dcol) = computer then inc n
    end if
  next d
  GetNumNeighbors = n
end function

' Returns 1 if a move is connected to an edge by friendly pieces.
' The connection must be a continuous straight line in one of the
' four orthagonal directions.
function IsConnectedToEdge(row, col)
  local trow, tcol, connected
  local cell, drow, dcol, index
  for index = 0 to SIZE-1
    connected = 1
    drow = offsets(index, 0)
    dcol = offsets(index, 1)
    trow = row
    tcol = col
    dok = 0
    do
      inc trow, drow
      inc tcol, dcol
      cell = board(trow, tcol)
      if cell = OUT then
        if connected then
          IsConnectedToEdge = 1
          exit function
        end if
      else if cell <> computer then
        connected = 0
        exit do
      end if      
    loop
  next index
  IsConnectedToEdge = 0
end function

' Return the 'policy' value of a proposed move
' This value is based on board weights and some
' state considerations for edge moves.
function GetPolicyValue(row, col, computer) as float
  local float v1, v2, v3
  v1 = policyWeight1*BoardWeights(row, col)
  v2 = policyWeight4*GetNumNeighbors(row, col)
  v3 = policyWeight5*IsConnectedToEdge(row, col)
  GetPolicyValue = v1 + v2 + v3
end function

' Evolve the board weights as the game progresses
sub UpdateBoardWeights
  local i, j, row, col, e
  local rrow, rcol
  local rb, rf, rs, cc, cb, cf, cs, rc
  local np, r1, c1, eweight
  ' detect endgame and set flag for lookahead
  np = GetNumPieces()
  'if np >= ENDGAME_NPIECES then
  '  isEndGame = 1
  '  SetEndgameWeights
  '  exit sub
  'end if
  for i = 0 to 3
    rrow = UpdateParams(i, 0)
    rcol = UpdateParams(i, 1)
    if board(rrow, rcol) = computer then
      r1 = UpdateParams(i, 2) : c1 = UpdateParams(i, 3)
      BoardWeights(r1, c1) = 10
      rb = UpdateParams(i, 4)
      rf = UpdateParams(i, 5)
      cc = UpdateParams(i, 6)
      rs = UpdateParams(i, 7)
      eweight = 10
      for j = rb to rf step rs
        if board(j, cc) = EMPTY then
          BoardWeights(j, cc) = eweight
          eweight = 0
        end if
      next j
      cb = UpdateParams(i, 8)
      cf = UpdateParams(i, 9)
      rc = UpdateParams(i, 10)
      cs = UpdateParams(i, 11)
      eweight = 10
      for j = cb to cf step cs
        if board(rc, j) = EMPTY then
          BoardWeights(rc, j) = eweight
          eweight = 0
        end if
      next j
    end if    
  next i
  for e = 1 to 4
    es = GetEdgeStatus(e)
    select case e
      case 1
        if es > 0 then
          for row = 1 to 8
            if board(row, 1) = EMPTY then inc BoardWeights(row, 1), 3
          next row
        else if es < 0 then
          for row = 1 to 8
            if board(row, 1) = EMPTY then inc BoardWeights(row, 1), -1
          next row
        end if 
      case 2
        if es > 0 then
          for col = 1 to 8
            if board(1, col) = EMPTY then inc BoardWeights(1, col), 3
          next col
        else if es < 0 then
          for col = 1 to 8
            if board(1, col) = EMPTY then inc BoardWeights(1, col), -1
          next col
        end if 
      case 3
        if es > 0 then
          for row = 1 to 8
            if board(row, 8) = EMPTY then inc BoardWeights(row, 8), 3
          next row
        else if es < 0 then
          for row = 1 to 8
            if board(row, 8) = EMPTY then inc BoardWeights(row, 8), -1
          next row
        end if 
      case 4
        if es > 0 then
          for col = 1 to 8
            if board(8, col) = EMPTY then inc BoardWeights(8, col), 3
          next col
        else if es < 0 then
          for col = 1 to 8
            if board(8, col) = EMPTY then inc BoardWeights(8, col), -1
          next col
        end if 
    end select
  next e        
  ShowPolicyDetails
end sub

' Get the status of the specified edge.
' Edges numbered 1-4, clock wise from left edge.
'  +N = occupied by N computer pieces only
'  -N = occupied by N player pieces only
'   0 = unoccupied
function GetEdgeStatus(which)
  local col, row, numc, nump, c1, c2, status
  numc = 0 : nump = 0 : c1 = 0 : c2 = 0
  select case which
    case 1
      for row = 1 to SIZE
        if board(row, 1) = computer then inc numc
        if board(row, 1) = player then inc nump
      next row
    case 2
      for col = 1 to SIZE
        if board(1, col) = computer then inc numc
        if board(1, col) = player then inc nump
      next col
    case 3
      for row = 1 to SIZE
        if board(row, 8) = computer then inc numc
        if board(row, 8) = player then inc nump
      next row
    case 4
      for col = 1 to SIZE
        if board(8, col) = computer then inc numc
        if board(8, col) = player then inc nump
      next col
  end select
 status = 0
  if numc > 0 and nump = 0 then status = numc
  if nump > 0 and numc = 0 then status = -nump
  GetEdgeStatus = status
end function

' In the end game, only pieces count using the lookahead
sub SetEndgameWeights
  policyWeight1 = 0
  policyWeight2 = 0
  policyWeight3 = 1
end sub

' =====================================================
' All the screen drawing methods are below this point
' =====================================================

' Draw the entire screen
sub DrawScreen
  local x, y
  cls
  x = LEFTMARGIN + SIZE*CELLSIZE\2
  y = TOPMARGIN - 30
  text x, y, "Reversi", "CB", 5,, rgb(yellow)
  DrawBoard
  if hasMouse then
    DrawMenu
  else
    DrawKBMenu
  end if
end sub
  
' Draw the board and the current pieces.
sub DrawBoard
  local row, col, x1, y1, x2, y2, ccs
  local bhs, bvs
  ccs = CELLSIZE
  bhs = SIZE*ccs : bvs = SIZE*CELLSIZE
  box LEFTMARGIN-BWIDTH, TOPMARGIN-BWIDTH, bhs+2*BWIDTH, bvs+2*BWIDTH,,, BCOLOR
  text LEFTMARGIN, TOPMARGIN-15, "  A    B    C    D    E    F    G    H",,,,, BCOLOR
  for i = 1 to 8
    text LEFTMARGIN-10, TOPMARGIN+15+(i-1)*CELLSIZE, str$(i),,,,,BCOLOR
  next i
  box LEFTMARGIN, TOPMARGIN, bhs, bvs,, CBLACK, BOARDCOLOR
  x1 = LEFTMARGIN
  x2 = x1 + SIZE*ccs
  for row = 1 to SIZE
    y1 = TOPMARGIN + row*CELLSIZE
    y2 = y1
    line x1, y1, x2, y2,, CBLACK
  next row
  y1 = TOPMARGIN
  for col = 1 to SIZE
    x1 = LEFTMARGIN + col*ccs
    x2 = x1
    line x1, y1, x2, y2,, CBLACK
  next col
  DrawPieces
end sub

'Draw all the pieces on the board
sub DrawPieces
  local row, col, cell
  for row = 1 to SIZE
    for col = 1 to SIZE
      cell = board(row, col)
      if cell = EMPTY then
        continue for
      end if
      DrawPiece row, col, cell
    next col
  next row
end sub

' Draw a playing piece at the specified cell
' for the specified player:
' Note: NO range checking is done on arguments!
sub DrawPiece row, col, cell
  local x, y, ccs
  local edgeColor, fillColor, thick
  if row <= 0 or row > SIZE or col <= 0 or col > SIZE then
    print
    print "DrawPiece: BUG -- coordinates out of range: " + str$(row) + "," + str$(col)
    close #1
    exit
  end if
  thick = 1
  edgeColor = 0
  fillColor = 0
  ccs = CELLSIZE
  if cell = PBLACK then
    edgeColor = CBLACK
    fillColor = CBLACK
  else
    edgeColor = CWHITE
    fillColor = CWHITE
  end if
  x = LEFTMARGIN + col*ccs - ccs/2
  y = TOPMARGIN + row*CELLSIZE - CELLSIZE/2
  circle x, y, PieceRadius, thick,, edgeColor, fillColor
end sub

' Hilite a piece by drawing a ring of contrasting color
' This is done to show the last move made.
sub HiliteLastMove row, col, cell
  local x, y, ccs
  local edgeColor, fillColor, thick
  if row = 0 or col = 0 then
    exit sub
  end if
  thick = 2
  edgeColor = 0
  fillColor = 0
  ccs = CELLSIZE
  if cell = PBLACK then
    edgeColor = CWHITE
    fillColor = CBLACK
  else
    edgeColor = CBLACK
    fillColor = CWHITE
  end if
  x = LEFTMARGIN + col*ccs - ccs/2
  y = TOPMARGIN + row*CELLSIZE - CELLSIZE/2
  circle x, y, PieceRadius, thick,, edgeColor, fillColor
end sub

' Show available moves for the specified player
sub ShowMoves who
  local availableMoves(20)
  local nmoves, i, move, row, col, mplayer, bookmark, c, ccs  
  if who = PBLACK then
    c = RGB(BLACK)
  else
    c = RGB(WHITE)
  end if
  ccs = CELLSIZE
  GetMoves who, availableMoves()
  nmoves = availableMoves(0)
  for i = 1 to nmoves
    move = availableMoves(i)  
    UnpackMove move, row, col, mplayer, bookmark
    x = LEFTMARGIN + col*ccs - ccs/2
    y = TOPMARGIN + row*CELLSIZE - CELLSIZE/2
    circle x, y, PieceRadius, 1,, c
  next i
end sub

' Show status messages below the board
sub DrawStatus message$
  local x, y. fx, fy
  x = STATUS_X : y = STATUS_Y
  box x, y, STATUS_WIDTH, STATUS_HEIGHT,, CBLACK, CBLACK
  fx = x + 10
  fy = y + 20
  text fx, fy, message$
end sub

' Draw The Menu (used by mouse only)
sub DrawMenu
  local x, xt
  for i = 1 to NMENUS
    x = MX+(i-1)*(MW+MSPAC)
    menuX(i) = x
    box x, MY, MW, MH
    tx = x+MW\2
    select case i
      case 1
        text tx, MY+5, "New", "CT"
      case 2
        text tx, MY+5, "Go", "CT"
      case 3
        text tx, MY+5, "Quit", "CT"
    end select
  next i
end sub

' Draw the Keyboard Menu
sub DrawKBMenu
  local x, y
  x = 500
  y = 0
  inc y, 15 : text x, y, "Move Cursor:    Arrow Keys"
  inc y, 15 : text x, y, "Move:           ENTER Key"
  inc y, 15 : text x, y, "New Game:       F12 Key"
  inc y, 15 : text x, y, "Help:           ? Key"
  inc y, 15 : text x, y, "Computer First: End Key"
  inc y, 15 : text x, y, "Quit:           Escape Key"
end sub

' Show the Help Screen
sub ShowHelp
  local z$
  cls
  gui cursor hide
  text mm.hres\2, 10, "Reversi Rules and Game Play", "CT", 4,, rgb(green)
  text 0, 40, ""
  print "Reversi (the commercial version is called 'Othello') is a board game in which players"
  print "take turns flipping two-side pieces upside down to show their color instead of the"
  print "opponent's color. Although a piece may be flipped many times during a game, pieces are"
  print "never moved. When the board fills up or neither player has a move, the game ends, and"
  print "the player with the most pieces showing his or her color wins. Ties are possible."
  print ""
  print "At the start of the game, 4 pieces, 2 of each color, are located in the 4 central squares"
  print "of the board. By default, the human player goes first, but the human can let the computer"
  print "go first by using the 'GO' command. (See commands below.) The first player always takes"
  print "the black pieces. The program will automatically show legal moves for the player"
  print "currently on the move by drawing open circles in that player's color."
  print ""
  print "A legal move is one that flips one or more of the opponent's pieces to make them your"
  print "pieces. In order to flip an opponent's piece, your move must put the opponent's piece or"
  print "pieces in a straight line between another of your pieces and the piece you are placing,"
  print "with no intervening empty spaces. Some moves can flip pieces in several directions at the"
  print "same time, depending on whether each of the 8 directions meets the criterion."
  print ""
  print "It should be obvious to you after a little thought that the 4 corner locations on the"
  print "board are very important, because once a piece is in the corner, it cannot be flipped,"
  print "since there is no way to put that piece between 2 of its enemy pieces."
  print ""
  print "Reversi can be played using a mouse or just the keyboard. To place a piece using the mouse,"
  print "just click the board square you want to place your piece into. If that square is not"
  print "marked showing it is a legal move, your click will be ignored."
  print ""
  print "To make a move using the keyboard, use the arrow keys on the keyboard to move the red"
  print "cursor to the board square you want to move into and then press the ENTER key."
  print ""
  print "Additional commands are available to start a new game, ask the computer to take the first"
  print "move, and to quit the program. For mouse users, click one of the menu buttons just below"
  print "the board. For keyboard users, type one of these keys:
  print "  F12 Key    -- New Game
  print "  End Key    -- Computer Goes first  (only at the start of a game)
  print "  Escape Key -- Quit the Program
  print "And the ? key can be used any time to show these instructions again."
  text mm.hres\2, 540, "Press Any Key to Continue", "CT"
  z$ = INKEY$
  do
    z$ = INKEY$
  loop until z$ <> ""
  gui cursor show
  DrawScreen
end sub

' ==================================================
' Only debugging methods and data below this point
' ==================================================

' Print the current board out textually
' For debugging only 
sub PrintBoard nrev
  local row, col
  print #1, "Number of pieces reversed: " + str$(nrev)
  print #1, "   1 2 3 4 5 6   8
  print #1, "  +-+-+-+-+-+-+-+-+
  for row = 1 to 8
  print #1, " " + str$(row) + "|";
  for col = 1 to 8
  select case board(row, col)
    case EMPTY
      print #1, " ";
    case PBLACK
      print #1, "B";
    case PWHITE
      print #1, "W";
  end select
  print #1, "|";
  next col
  print #1, ""
  print #1, "  +-+-+-+-+-+-+-+-+
  next row
end sub

' For debugging only
sub ShowPolicyDetails
  local int row, col, x, y, w, vl
  local string out$, val$
  x = POLICY_X
  y = POLICY_Y
  text x, y, "Board Weights"
  y = y + 12
  for row = 1 to SIZE
    out$ = ""
    for col = 1 to SIZE 
      w = BoardWeights(row, col)
      if w = OCCUPIED then
        val$ = "."
      else      
        val$ = str$(w)
      end if
      vl = LEN(val$)
      out$ = out$ + val$ + SPACE$(4-vl)
    next col
    text x, y, out$
    y = y + 12
  next row 
end sub

' Build a list of initial pieces for debugging
sub BuildPieceList
  local i, row, col, player, auto
  local f1$, f2$, f3$
  cls
  print "Row and Column range is 1-8; Player is 1(Black) or -1(white)"
  print "Enter Zero for Row to finish"
  print "Enter -1 to use canned cells"
  print ""
  ZeroBoard
  auto = 0
  do
    input "Enter Row, Col, Player: ", f1$, f2$, f3$
    row = val(f1$) : col = val(f2$) : player = val(f3$)
    if row > 0 then      
      if row < 1 or row > 8 then exit do
      if col < 1 or col > 8 then exit do
      if player <> 1 and player <> -1 then exit do
      board(row, col) = player
    else
      auto = 1
      exit do
    end if
  loop
  if auto then
    for row = 1 to SIZE
      for col = 1 to SIZE
        board(row, col) = PBLACK
      next col
    next row
    board(7, 7) = EMPTY
    board(5, 5) = PWHITE
    onTheMove = -1
  end if
end sub

' Debugging
' Load the board pieces from a file
' pcolor is player color, mover is who is on the move
sub LoadBoard fname$, pcolor, mover
  local row, col
  local buf$, v$, who
  ZeroBoard
  open fname$ for input as #2
  for row = 1 to SIZE
    line input #2, buf$
    buf$ = buf$ + ", "
    for col = 1 to SIZE
      v$ = FIELD$(buf$, col, ",")
      board(row, col) = val(v$)
    next col
  next row
  close #2
  player = pcolor
  onTheMove = mover
end sub

' Initial Board Weights
weights:
data  99, -50, 3,   5,   5, 3, -50,  99
data -50, -99, 4,   4,   4, 4, -99, -50
data   3,   4, 7,   4,   4, 7,   4,   3
data   5,   1, 4,  17,  17, 4,   1,   5
data   5,   1, 4,  17,  17, 4,   1,   5
data   3,   4, 7,   4,   4, 7,   4,   3
data -50, -99, 4,   1,   1, 4, -99, -50
data  99, -50, 3,   5,   5, 3, -50,  99

' Update Scan specificatons
data  1, 1, 2, 2, 1, 7, 1, 1, 1, 7, 1, 1
data  1, 8, 2, 7, 1, 7, 8, 1, 7, 1, 1,-1
data  8, 1, 7, 2, 7, 1, 1,-1, 1, 7, 8, 1
data  8, 8, 7, 7, 7, 2, 8,-1, 7, 1, 8,-1

